<?php
if ( ! defined( 'THEME_FRAMEWORK' ) ) {
	exit( 'No direct script access allowed' );
}

/**
 * Helper functions for logic part of control panel
 *
 * @author         Reza Marandi <ross@artbees.net>
 * @copyright   Artbees LTD (c)
 * @link        http://artbees.net
 * @package     artbees
 */
class Abb_Logic_Helpers {

	/**
	 * method that is resposible to unzip compress files .
	 * it used native wordpress functions.
	 *
	 * @since       1.0.0
	 * @author Reza Marandi <ross@artbees.net>
	 *
	 * @param str $zip_path compress file absolute path.
	 * @param str $dest_path Where should it be uncompressed.
	 *
	 * @return bool will return boolean status of action
	 */
	static function unZip( $zip_path, $dest_path ) {

		global $wp_filesystem;
		
		$zip_path  = realpath( $zip_path );
		$dest_path = realpath( $dest_path );

		$url = wp_nonce_url('admin.php?page=theme-templates','mk-check-ftp-credentials');
		
		$wp_filesystem = self::get_fs_credentials( $url );
		
		if( is_wp_error( $wp_filesystem ) ){
			throw new Exception( $wp_filesystem->get_error_message() );
			return false;
		}

		if ( !$wp_filesystem->exists( $zip_path ) ) {
			throw new Exception( __( 'Zip file that you are looking for is not exist' , 'mk_framework' ) );
			return false;
		}

		if ( !$wp_filesystem->is_writable( $dest_path ) ) {
			throw new Exception( __( 'Destination path is not writable , Please resolve this issue first.' , 'mk_framework' ) );
			return false;
		}

		$unzipfile = unzip_file( $zip_path, $dest_path );
		if ( is_wp_error( $unzipfile ) ) {
			throw new Exception( $unzipfile->get_error_message(), 1 );
			return false;
		} else {
			return true;
		}
	}
	/**
	 * You can create a directory using this helper , it will check the dest directory for if its writable or not then
	 * try to create new one
	 *
	 * @since       1.0.0
	 * @author Reza Marandi <ross@artbees.net>
	 *
	 * @param str $path path of directory that need to be created
	 * @param int $perm permission of new directory , default is : 0775
	 *
	 *     @return bool will return boolean status of action , all message is setted to $this->message()
	 */
	static function checkPermAndCreate( $path, $perm = 0775 ) {

		global $wp_filesystem;

		$url = wp_nonce_url('admin.php?page=theme-templates','mk-check-ftp-credentials');
		
		$wp_filesystem = self::get_fs_credentials( $url );
		
		if( is_wp_error( $wp_filesystem ) ){
			throw new Exception( $wp_filesystem->get_error_message() );
			return false;
		}

		if( $wp_filesystem->exists( $path ) ){
			if( !$wp_filesystem->is_writable( $path ) ){
				throw new Exception( sprintf( __( '%s directory is not writable', 'mk_framework' ) , $path ) );
				return false;
			}
			return true;
		}else{
			if( !$wp_filesystem->mkdir( $path, $perm ) ){
				throw new Exception( sprintf( __( 'Can\'t create directory %s', 'mk_framework' ) , $path ) );
				return false;
			}
			return true;
		}
	}
	/**
	 * this method is resposible to download file from url and save it on server.
	 * it will check if curl is available or not and then decide to use curl or file_get_content
	 *
	 * @since       1.0.0
	 * @author Reza Marandi <ross@artbees.net>
	 *
	 * @param string $url url of file (http://yahoo.com/test-plugin.zip).
	 * @param string $file_name name of the fire that should be create at destination directory.
	 * @param string $dest_directory absolute path of directory that file save on it.
	 *
	 * @return bool will return action status
	 */
	static function uploadFromURL( $url, $file_name, $dest_directory ) {

		global $wp_filesystem;

		set_time_limit( 0 );

		try {
			self::checkPermAndCreate( $dest_directory );
		} catch (Exception $e) {
			throw new Exception( sprintf( __( 'Destination directory is not ready for upload . {%s}',  'mk_framework' ) , $dest_directory ) );
			return false;
		}

		$response = wp_remote_get( $url );

		if( is_wp_error( $response ) ) {
			throw new Exception( $response->get_error_message() );
			return false;
		}

		$response_body = wp_remote_retrieve_body( $response );

		// Check for error
		if ( is_wp_error( $response_body ) ) {
			throw new Exception( $response_body->get_error_message() );
			return false;
		}

		$url = wp_nonce_url('admin.php?page=theme-templates','mk-check-ftp-credentials');

		$wp_filesystem = Abb_Logic_Helpers::get_fs_credentials( $url );

		if( is_wp_error( $wp_filesystem ) ){
			throw new Exception( $wp_filesystem->errors->get_error_message() );
			return false;
		}

		if( !$wp_filesystem->put_contents( $dest_directory . $file_name, $response_body ) ){
			throw new Exception( sprintf( __( "Can't write file to {%s}", 'mk_framework' ) , $dest_directory . $file_name ) );
			return false;
		}

		return $dest_directory . $file_name;
	}


	/**
	 * this method is resposible to check a directory for see if its writebale or not
	 *
	 * @since       1.0.0
	 * @author Reza Marandi <ross@artbees.net>
	 *
	 * @param str $path for example (/var/www/jupiter/wp-content/plugins)
	 *
	 * @return bool true or false
	 */
	static function writableOwnerCheck( $path ) {
		$is_writable = is_writable( $path );
		@file_put_contents( realpath( $path ) . '/testFile', 'test' );
		if ( file_exists( realpath( $path ) . '/testFile' ) == false ) {
			return false;
		}
		$created_file_owner = fileowner( realpath( $path ) . '/testFile' );
		unlink( preg_replace( '/([^:])(\/{2,})/', '$1/', $path . '/testFile' ) );
		$path_owner = fileowner( $path );
		if ( $is_writable == true && $created_file_owner == $path_owner ) {
			return true;
		}
		return false;
	}
	/**
	 * this method is resposible to delete a directory or file
	 * if the path is pointing to a directory it will remove all the includes file recursivly and then remove directory at last step
	 * if the path is pointing to a file it will remove it
	 *
	 * @since       1.0.0
	 * @author Reza Marandi <ross@artbees.net>
	 *
	 * @param str $dir for example (/var/www/jupiter/wp-content/plugins)
	 *
	 * @return bool true or false
	 */
	static function deleteFileNDir( $dir ) {

		global $wp_filesystem;

		if ( empty( $dir ) == true || strlen( $dir ) < 2 ) {
			return false;
		}

		$dir = realpath( $dir );

		if ( ! $wp_filesystem->exists( $dir ) ) {
			return true;
		}

		if ( $wp_filesystem->is_dir( $dir ) ) {
			return $wp_filesystem->rmdir( $dir, true );
		}else{
			return $wp_filesystem->delete( $dir );
		}

	}
	/**
	 * Safely and securely get file from server.
	 * It attempts to read file using Wordpress native file read functions
	 * If it fails, we use wp_remote_get. if the site is ssl enabled, we try to convert it http as some servers may fail to get file
	 *
	 * @author Reza Marandi <ross@artbees.net>
	 *
	 * @param $file_url         string    its directory URL
	 * @param $file_dir         string    its directory Path
	 *
	 * @return $wp_file_body    string
	 */
	static function getFileBody( $file_uri, $file_dir ) {
		global $wp_filesystem;

		$file_dir = realpath( $file_dir );


		$url = wp_nonce_url('admin.php?page=theme-templates','mk-check-ftp-credentials');
		
		$wp_filesystem = self::get_fs_credentials( $url );
		
		if( is_wp_error( $wp_filesystem ) ){
			throw new Exception( $wp_filesystem->get_error_message() );
			return false;
		}

		$wp_get_file_body = $wp_filesystem->get_contents( $file_dir );
		if ( $wp_get_file_body == false ) {
			$wp_remote_get_file = wp_remote_get( $file_uri );

			if ( is_array( $wp_remote_get_file ) and array_key_exists( 'body', $wp_remote_get_file ) ) {
				$wp_remote_get_file_body = $wp_remote_get_file['body'];

			} else if ( is_numeric( strpos( $file_uri, 'https://' ) ) ) {

				$file_uri           = str_replace( 'https://', 'http://', $file_uri );
				$wp_remote_get_file = wp_remote_get( $file_uri );

				if ( ! is_array( $wp_remote_get_file ) or ! array_key_exists( 'body', $wp_remote_get_file ) ) {
					throw new Exception( __( 'SSL connection error. Code: template-assets-get','mk_framework' ) );
					return false;
				}

				$wp_remote_get_file_body = $wp_remote_get_file['body'];
			}

			$wp_file_body = $wp_remote_get_file_body;

		} else {
			$wp_file_body = $wp_get_file_body;
		}
		return $wp_file_body;
	}
	/**
	 * It will check the header of URL and return boolean.
	 * False if header is 404
	 * True if header is something else
	 *
	 * @author Reza Marandi <ross@artbees.net>
	 *
	 * @param string $url string of url for checking
	 *
	 * @return boolean true if header is not 404
	 */
	static function remoteURLHeaderCheck( $url ) {
		if ( strpos( @get_headers( $url )[0] , '404 Not Found' ) == false ) {
			return true;
		} else {
			return false;
		}
	}

	/**
	 * It will create a compress file from list of files
	 *
	 * @author Reza Marandi <ross@artbees.net>
	 *
	 * @param array   $files for example : array('preload-images/5.jpg','kwicks/ringo.gif','rod.jpg','reddit.gif');
	 * @param string  $destination name of the file or full address of destination for example : my-archive.zip
	 * @param boolean $overwrite if destionation exist , should it overwrite the compress file ?
	 *
	 * @return boolean true if completed and false if something goes wrong
	 */
	static function zip( $files = array(), $destination = '', $overwrite = false ) {

		global $wp_filesystem;

		$url = wp_nonce_url('admin.php?page=theme-templates','mk-check-ftp-credentials');
		
		$wp_filesystem = self::get_fs_credentials( $url );
		
		if( is_wp_error( $wp_filesystem ) ){
			throw new Exception( $wp_filesystem->get_error_message() );
			return false;
		}

		// if the zip file already exists and overwrite is false, return false
		if ( $wp_filesystem->exists($destination) && !$overwrite ){
			return false;
		}

		// vars
		$valid_files = array();

		// if files were passed in...
		if ( is_array( $files ) ) {
			// cycle through each file
			foreach ( $files as $file_name => $file_path ) {
				// make sure the file exists
				if ( $wp_filesystem->exists( $file_path ) ) {
					$valid_files[$file_name] = $file_path;
				}
			}
		}
		// if we have good files...
		if ( count( $valid_files ) ) {

			$temp_file = tempnam( sys_get_temp_dir(), 'zip' );

			$zip = new ZipArchive();

			// Try open the temp file.
			$zip->open( $temp_file );

			// add the files to archive.
			foreach ( $valid_files as $file_name => $file_path ) {
				$zip->addFile( $file_path, $file_name );
			}

			// debug
			// echo 'The zip archive contains ',$zip->numFiles,' files with a status of ',$zip->status;
			// close the zip -- done!
			$zip->close();

			// Copy the temp file to destination.
			$wp_filesystem->copy( $temp_file, $destination, true, 0644 );

			// Try delete the temp file.
			@$wp_filesystem->delete( $temp_file );

			// check to make sure the file exists.
			return $wp_filesystem->exists(  $destination );

		} else {
			return false;
		}
	}
	static function search_multdim( $array, $key, $value ) {
		return (array_search( $value, array_column( $array, $key ) ));
	}
	/**
	 * It will check wether wordpress-importer plugin is exist in plugin directory or not.
	 * if exist it will return the wordpress importer file
	 * if not it will use jupiter version
	 *
	 * @author      Reza Marandi <ross@artbees.net>
	 * @copyright   Artbees LTD (c)
	 * @link        http://artbees.net
	 * @since       Version 5.5
	 */

	static function include_wordpress_importer() {

		if ( class_exists( 'WP_Import' ) === true ) {
			return true;
		}

		if ( is_plugin_active( 'wordpress-importer' ) ) {
			$plugins_data = get_plugins();
			$result = preg_grep( "/\bwordpress-importer\b/i", array_keys( $plugins_data ) );
			if ( is_array( $result ) && count( $result ) > 0 ) {
				$keys = array_keys( $plugins_data );
				include WP_PLUGIN_DIR . '/' . $keys[ key( $result ) ];
				return true;
			}
		}

		include THEME_CONTROL_PANEL . '/logic/wordpress-importer.php';
		return true;
	}
	/**
	 * It will return permission of directory
	 *
	 * @author Reza Marandi <ross@artbees.net>
	 *
	 * @param string $path Full path of directory
	 *
	 * @return int
	 */
	static function get_perm( $path ) {
		return substr( sprintf( '%o', fileperms( ABSPATH . $path ) ), -4 );
	}

	/**
	 * Get filesytem credentials
	 *
	 * @author Sofyan Sitorus <sofyan@artbees.net>
	 *
	 * @param string $url URL for the Form credentials
	 *
	 * @param boolean $print_form Print the form to output
	 *
	 * @return int
	 */
	static function get_fs_credentials( $form_post = '', $type = '', $error = false, $context = '', $extra_fields = null, $allow_relaxed_file_ownership = false, $print_form = false ) {

		global $wp_filesystem;

		require_once ABSPATH . '/wp-admin/includes/file.php';
		require_once ABSPATH . '/wp-admin/includes/template.php';
		require_once THEME_CONTROL_PANEL . "/logic/system-messages/logic-messages-lib.php";

		if( !$print_form ){
			ob_start();
		}

		$creds = request_filesystem_credentials( $form_post, $type, $error, $context, $extra_fields, $allow_relaxed_file_ownership );

		if( !$print_form ){
			ob_end_clean();
		}
		
		if ( false === $creds ){
			return new WP_Error( 'no_filesystem_credentials', mk_logic_message_helper( 'filesystem', 'No filesystem credentials found!' ) );
		}

		if ( !WP_Filesystem( $creds, $context, $allow_relaxed_file_ownership ) ) {
			return $wp_filesystem->errors;
		}

		return $wp_filesystem;
	}
}
